import React from 'react'
import { sfx } from '../utils/audio'
import { motion, AnimatePresence } from 'framer-motion'

export default function ColorTestIshihara({ onDone, onCancel, images }) {
  const plates = images && images.length ? images : []
  const [idx, setIdx] = React.useState(0)
  const [answers, setAnswers] = React.useState([])
  const [input, setInput] = React.useState('')

  const plate = plates[idx]

  const handleNum = (num) => {
    if (input.length < 2) {
      sfx.play('click')
      setInput(prev => prev + num)
    }
  }

  const handleDel = () => {
    sfx.play('click')
    setInput(prev => prev.slice(0, -1))
  }

  const handleQuestion = () => {
    sfx.play('click')
    setInput('?')
    setTimeout(() => submit('?'), 300)
  }

  const submit = (val) => {
    const finalVal = val !== undefined ? val : input
    if (!finalVal) return

    const rec = { plateId: plate.id, value: finalVal }
    const newAnswers = [...answers, rec]
    setAnswers(newAnswers)
    setInput('')

    if (idx < plates.length - 1) {
      sfx.play('click')
      setIdx(i => i + 1)
    } else {
      sfx.play('success')
      onDone?.(newAnswers)
    }
  }

  if (!plate) return null

  return (
    <div className="fixed inset-0 z-50 flex items-center justify-center bg-slate-900/90 backdrop-blur-sm p-4">
      <div className="w-full max-w-4xl bg-white rounded-3xl shadow-2xl overflow-hidden flex flex-col md:flex-row h-auto max-h-[90vh]">

        {/* Left: Image Display */}
        <div className="flex-1 bg-slate-100 relative flex flex-col items-center justify-center p-6">
          <div className="absolute top-4 left-4 bg-white px-4 py-2 rounded-full shadow-sm font-bold text-slate-600">
            Plate {idx + 1} / {plates.length}
          </div>
          <button
            onClick={onCancel}
            className="absolute top-4 right-4 text-slate-400 hover:text-red-500 transition-colors"
          >
            <span className="text-2xl">✕</span>
          </button>

          <AnimatePresence mode='wait'>
            <motion.img
              key={plate.id}
              initial={{ opacity: 0, scale: 0.8 }}
              animate={{ opacity: 1, scale: 1 }}
              exit={{ opacity: 0, scale: 1.1 }}
              transition={{ type: 'spring', damping: 20 }}
              src={plate.url}
              alt={`Plate ${plate.id}`}
              className="w-full max-w-md object-contain shadow-lg rounded-full border-4 border-white"
            />
          </AnimatePresence>

          <p className="mt-6 text-slate-500 text-center font-medium">
            Apa angka yang kamu lihat?
          </p>
        </div>

        {/* Right: Numpad Control */}
        <div className="w-full md:w-96 bg-white p-6 flex flex-col overflow-y-auto">
          <div className="flex-1 flex flex-col justify-center">
            {/* Display Input */}
            <div className="bg-slate-100 rounded-2xl h-20 mb-6 flex items-center justify-center border-2 border-slate-200">
              <span className="text-4xl font-bold text-slate-700 tracking-widest">
                {input || <span className="text-slate-300">...</span>}
              </span>
            </div>

            {/* Numpad Grid */}
            <div className="grid grid-cols-3 gap-3">
              {[1, 2, 3, 4, 5, 6, 7, 8, 9].map(n => (
                <button
                  key={n}
                  onClick={() => handleNum(n)}
                  className="aspect-square rounded-xl bg-slate-50 border-b-4 border-slate-200 active:border-b-0 active:translate-y-1 hover:bg-indigo-50 hover:text-indigo-600 text-2xl font-bold text-slate-600 transition-all"
                >
                  {n}
                </button>
              ))}
              <button
                onClick={handleDel}
                className="aspect-square rounded-xl bg-red-50 border-b-4 border-red-200 active:border-b-0 active:translate-y-1 text-red-500 font-bold text-xl transition-all flex items-center justify-center"
              >
                ⌫
              </button>
              <button
                onClick={() => handleNum(0)}
                className="aspect-square rounded-xl bg-slate-50 border-b-4 border-slate-200 active:border-b-0 active:translate-y-1 hover:bg-indigo-50 hover:text-indigo-600 text-2xl font-bold text-slate-600 transition-all"
              >
                0
              </button>
              <button
                onClick={handleQuestion}
                className="aspect-square rounded-xl bg-yellow-50 border-b-4 border-yellow-200 active:border-b-0 active:translate-y-1 text-yellow-600 font-bold text-xl transition-all flex flex-col items-center justify-center leading-none"
              >
                <span>?</span>
                <span className="text-[10px] mt-1">Tidak Tahu</span>
              </button>
            </div>
          </div>

          <button
            onClick={() => submit()}
            disabled={!input}
            className={`w-full mt-6 py-4 rounded-xl font-bold text-lg shadow-lg transition-all ${input
              ? 'bg-gradient-to-r from-indigo-500 to-purple-600 text-white hover:shadow-xl hover:scale-[1.02] active:scale-[0.98]'
              : 'bg-slate-200 text-slate-400 cursor-not-allowed'
              }`}
          >
            Jawab
          </button>
        </div>
      </div>
    </div>
  )
}
