import React from 'react'
import { useLocation, useNavigate } from 'react-router-dom'
import { sfx } from '../utils/audio'
import { register as registerLocal, loginWithEmail as loginLocal, loginWithGoogle as loginGoogleLocal } from '../api/auth'
import * as serverAuth from '../api/serverAuth'

export default function Auth() {
  const [role, setRoleState] = React.useState('student')
  const [email, setEmail] = React.useState('')
  const [username, setUsername] = React.useState('')
  const [password, setPassword] = React.useState('')
  const [name, setName] = React.useState('')
  const [mode, setMode] = React.useState('login') // 'login' | 'register'
  const [error, setError] = React.useState('')
  const navigate = useNavigate()
  const location = useLocation()
  const from = location.state?.from || '/'
  const useServer = Boolean(import.meta.env.VITE_API_BASE)

  React.useEffect(() => {
    const cid = import.meta.env.VITE_GOOGLE_CLIENT_ID
    if (role === 'student' && window.google && cid) {
      try {
        window.google.accounts.id.initialize({
          client_id: cid,
          callback: (resp) => {
            // Note: Untuk produksi, verifikasi credential JWT di backend untuk mendapat email/nama.
            const uEmail = email || 'googleuser@example.com'
            const uName = name || 'Siswa'
              ; (async () => {
                try {
                  if (useServer) { await serverAuth.loginGoogle({ email: uEmail, name: uName }) }
                  else { loginGoogleLocal({ email: uEmail, name: uName }) }
                  sfx.play('success'); navigate(from, { replace: true })
                } catch (err) { console.error(err) }
              })()
          },
        })
        const btn = document.getElementById('googleBtn')
        if (btn) {
          window.google.accounts.id.renderButton(btn, { theme: 'outline', size: 'large', shape: 'pill', text: 'continue_with' })
        }
      } catch { }
    }
  }, [role])

  const loginEmail = async (e) => {
    e.preventDefault(); setError('')
    const tEmail = (email || '').trim()
    const tPass = (password || '').trim()
    const tUsername = (username || '').trim()

    if (!tEmail || !tPass) { setError('Mohon isi Username/Email dan Password'); return }
    try {
      if (mode === 'register') {
        const regName = (name || (tEmail.split('@')[0]) || (role === 'admin' ? 'Guru SD' : 'Siswa SD')).trim()
        if (!regName || !tUsername) { setError('Nama dan Username wajib diisi'); return }
        if (useServer) await serverAuth.register({ name: regName, email: tEmail, username: tUsername, password: tPass, role })
        else registerLocal({ name: regName, email: tEmail, password: tPass, role })
      } else {
        // For login, tEmail acts as identifier (email or username)
        if (useServer) await serverAuth.login({ identifier: tEmail, password: tPass })
        else loginLocal({ email: tEmail, password: tPass })
      }
      sfx.play('success'); navigate(from, { replace: true })
    } catch (err) {
      setError(err.message || 'Gagal autentikasi')
    }
  }

  const loginGoogle = async () => {
    const cid = import.meta.env.VITE_GOOGLE_CLIENT_ID
    if (window.google && cid) {
      window.google.accounts.id.prompt()
    } else {
      const uEmail = email || 'student@example.com'
      const uName = name || 'Siswa (Mock)'
      try {
        if (useServer) { await serverAuth.loginGoogle({ email: uEmail, name: uName }) }
        else { loginGoogleLocal({ email: uEmail, name: uName }) }
        sfx.play('success'); navigate(from, { replace: true })
      }
      catch (err) { setError(err.message || 'Gagal login Google') }
    }
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-indigo-500 via-purple-500 to-pink-500 flex items-center justify-center px-4 py-10">
      <div className="max-w-md w-full mx-auto bg-white/95 backdrop-blur-xl border border-white/50 rounded-3xl p-8 shadow-2xl">
        <div className="text-center mb-8">
          <h1 className="heading text-3xl text-transparent bg-clip-text bg-gradient-to-r from-indigo-600 to-purple-600 mb-2">
            {mode === 'login' ? 'Selamat Datang!' : 'Buat Akun Baru'}
          </h1>
          <p className="text-slate-500">
            {mode === 'login' ? 'Masuk untuk melanjutkan petualanganmu' : 'Mulai perjalanan belajarmu hari ini'}
          </p>
        </div>

        <div className="flex bg-slate-100 p-1 rounded-full mb-6">
          <button
            onClick={() => { setMode('login'); setError('') }}
            className={`flex-1 rounded-full py-2 text-sm font-bold transition-all duration-300 ${mode === 'login' ? 'bg-white text-indigo-600 shadow-sm' : 'text-slate-500 hover:text-indigo-600'}`}
          >
            Masuk
          </button>
          <button
            onClick={() => { setMode('register'); setError('') }}
            className={`flex-1 rounded-full py-2 text-sm font-bold transition-all duration-300 ${mode === 'register' ? 'bg-white text-indigo-600 shadow-sm' : 'text-slate-500 hover:text-indigo-600'}`}
          >
            Daftar
          </button>
        </div>

        <div className="mb-6">
          <span className="block text-xs font-bold text-slate-400 uppercase tracking-wider mb-2 text-center">Login Sebagai</span>
          <div className="grid grid-cols-2 gap-3">
            <button
              onClick={() => setRoleState('student')}
              className={`rounded-xl px-4 py-3 border-2 font-bold transition-all ${role === 'student' ? 'border-indigo-500 bg-indigo-50 text-indigo-700' : 'border-slate-200 text-slate-500 hover:border-indigo-200'}`}
            >
              🎓 Siswa
            </button>
            <button
              onClick={() => setRoleState('admin')}
              className={`rounded-xl px-4 py-3 border-2 font-bold transition-all ${role === 'admin' ? 'border-indigo-500 bg-indigo-50 text-indigo-700' : 'border-slate-200 text-slate-500 hover:border-indigo-200'}`}
            >
              👨‍🏫 Guru
            </button>
          </div>
        </div>

        {role === 'student' && (
          <div className="space-y-3 mb-6">
            <div id="googleBtn" className="flex justify-center min-h-[40px]"></div>
            <div className="relative flex items-center gap-2 py-2">
              <div className="h-px bg-slate-200 flex-1"></div>
              <span className="text-xs text-slate-400 font-medium">ATAU</span>
              <div className="h-px bg-slate-200 flex-1"></div>
            </div>
          </div>
        )}

        <form onSubmit={loginEmail} className="space-y-4">
          {mode === 'register' && (
            <>
              <div className="space-y-1">
                <label className="text-sm font-bold text-slate-700 ml-1">Nama Lengkap</label>
                <input
                  value={name}
                  onChange={e => setName(e.target.value)}
                  type="text"
                  required
                  className="w-full bg-slate-50 border-2 border-slate-200 rounded-xl p-3 focus:border-indigo-500 focus:bg-white outline-none transition-colors"
                  placeholder="Contoh: Budi Santoso"
                />
              </div>
              <div className="space-y-1">
                <label className="text-sm font-bold text-slate-700 ml-1">Username</label>
                <input
                  value={username}
                  onChange={e => setUsername(e.target.value)}
                  type="text"
                  required
                  className="w-full bg-slate-50 border-2 border-slate-200 rounded-xl p-3 focus:border-indigo-500 focus:bg-white outline-none transition-colors"
                  placeholder="Username unik"
                />
              </div>
            </>
          )}
          <div className="space-y-1">
            <label className="text-sm font-bold text-slate-700 ml-1">{mode === 'login' ? 'Username / Email' : 'Email'}</label>
            <input
              value={email}
              onChange={e => setEmail(e.target.value)}
              type="text"
              required
              className="w-full bg-slate-50 border-2 border-slate-200 rounded-xl p-3 focus:border-indigo-500 focus:bg-white outline-none transition-colors"
              placeholder={mode === 'login' ? 'Username atau Email' : 'nama@email.com'}
            />
          </div>
          <div className="space-y-1">
            <label className="text-sm font-bold text-slate-700 ml-1">Password</label>
            <input
              value={password}
              onChange={e => setPassword(e.target.value)}
              type="password"
              required
              className="w-full bg-slate-50 border-2 border-slate-200 rounded-xl p-3 focus:border-indigo-500 focus:bg-white outline-none transition-colors"
              placeholder="••••••••"
            />
          </div>

          {error && (
            <div className="bg-red-50 text-red-600 text-sm p-3 rounded-xl flex items-center gap-2">
              <span>⚠️</span> {error}
            </div>
          )}

          <button className="w-full rounded-xl bg-gradient-to-r from-indigo-600 to-purple-600 text-white font-bold py-3.5 shadow-lg hover:shadow-xl hover:scale-[1.02] active:scale-[0.98] transition-all duration-200">
            {mode === 'register' ? 'Daftar Sekarang' : 'Masuk ke Akun'}
          </button>
        </form>
      </div>
    </div>
  )
}
