<!DOCTYPE html>
<html lang="id">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>Misi Bilangan Bulat: Angkasa Luar</title>
    <style>
        body {
            margin: 0;
            overflow: hidden;
            background-color: #0f172a;
            /* Slate 900 */
            font-family: 'Arial', sans-serif;
            touch-action: none;
            /* Mencegah zoom/scroll di mobile */
        }

        canvas {
            display: block;
        }

        .btn-back {
            position: absolute;
            top: 10px;
            right: 10px;
            z-index: 100;
            background-color: #ef4444;
            /* Red-500 */
            color: white;
            padding: 8px 16px;
            text-decoration: none;
            border-radius: 8px;
            font-family: 'Arial', sans-serif;
            font-weight: bold;
            border: 2px solid rgba(255, 255, 255, 0.2);
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.3);
            transition: all 0.2s ease;
        }

        .btn-back:hover {
            background-color: #dc2626;
            /* Red-600 */
            transform: translateY(-2px);
            box-shadow: 0 6px 8px rgba(0, 0, 0, 0.4);
        }

        .btn-back:active {
            transform: translateY(0);
        }
    </style>
</head>

<body>
<!-- Back Button -->
    <a href="index.php?game_id=2" class="btn-back">
        KEMBALI
    </a>

    <!-- Tutorial Overlay (How to Play) -->
    <div id="tutorial-overlay" style="
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(15, 23, 42, 0.95);
        backdrop-filter: blur(8px);
        z-index: 200;
        display: flex;
        flex-direction: column;
        justify-content: center;
        align-items: center;
        text-align: center;
        padding: 20px;
        box-sizing: border-box;
    ">
        <div style="
            background: rgba(30, 41, 59, 0.9);
            border: 2px solid #22d3ee;
            border-radius: 16px;
            padding: 30px;
            max-width: 450px;
            box-shadow: 0 0 30px rgba(34, 211, 238, 0.3);
        ">
            <h1 style="color: #22d3ee; font-size: 28px; margin-bottom: 15px;">🚀 Misi Bilangan Bulat</h1>
            <p style="color: #94a3b8; font-size: 18px; margin-bottom: 20px;">Tangkap asteroid dengan jawaban yang benar!</p>
            
            <div style="text-align: left; color: #e2e8f0; font-size: 16px; line-height: 1.8; margin-bottom: 25px;">
                <p>⬅️ ➡️ <strong>Panah / A D</strong> - Gerak kiri/kanan</p>
                <p>🖱️ <strong>Mouse/Touch</strong> - Ikuti jari/kursor</p>
                <p>⏬ <strong>SPASI (tahan)</strong> - Asteroid jatuh cepat</p>
                <p>🎯 <strong>Tujuan</strong> - Kumpulkan 150 poin!</p>
            </div>
            
            <button id="start-game-btn" style="
                background: linear-gradient(135deg, #22d3ee, #06b6d4);
                color: #0f172a;
                font-family: Arial, sans-serif;
                font-size: 20px;
                font-weight: bold;
                padding: 15px 40px;
                border: none;
                border-radius: 10px;
                cursor: pointer;
                transition: all 0.2s ease;
                box-shadow: 0 4px 15px rgba(34, 211, 238, 0.4);
            ">
                🎮 MULAI MAIN
            </button>
        </div>
    </div>

    <canvas id="gameCanvas"></canvas>

    <script>
        // ==========================================
        // 1. SETUP & VARIABEL GLOBAL
        // ==========================================
        const canvas = document.getElementById('gameCanvas');
        const ctx = canvas.getContext('2d');

        let width, height;
        let particles = [];
        let asteroids = [];
        let ship;
        let score = 0;
        let lives = 3;
        let level = 1;
        let gameState = "START"; // START, PLAYING, GAMEOVER, VICTORY
        let currentQuestion = null;
        let dbQuestions = {}; // Store fetched questions

        // Input State
        let keys = { left: false, right: false, space: false };
        let touchX = null;

        // ==========================================
        // 2. INPUT HANDLING (Keyboard & Touch)
        // ==========================================
        window.addEventListener('resize', resize);

        function resize() {
            width = canvas.width = window.innerWidth;
            height = canvas.height = window.innerHeight;
        }
        resize();

        // Keyboard
        window.addEventListener('keydown', e => {
            if (e.key === 'ArrowLeft' || e.key === 'a' || e.key === 'A') keys.left = true;
            if (e.key === 'ArrowRight' || e.key === 'd' || e.key === 'D') keys.right = true;
            if (e.code === 'Space') keys.space = true; // Tombol Spasi
        });

        window.addEventListener('keyup', e => {
            if (e.key === 'ArrowLeft' || e.key === 'a' || e.key === 'A') keys.left = false;
            if (e.key === 'ArrowRight' || e.key === 'd' || e.key === 'D') keys.right = false;
            if (e.code === 'Space') keys.space = false;
        });

        // Touch / Mouse untuk Menu
        window.addEventListener('mousedown', handleInputStart);
        window.addEventListener('touchstart', handleInputStart, { passive: false });

        function handleInputStart(e) {
            // Klik untuk Restart saat Game Over atau Menang
            if (gameState === "START" || gameState === "GAMEOVER" || gameState === "VICTORY") {
                resetGame();
            } else if (gameState === "PLAYING") {
                // Kontrol Touch untuk pergerakan pesawat
                let clientX = e.touches ? e.touches[0].clientX : e.clientX;
                touchX = clientX;
            }
        }

        window.addEventListener('mousemove', e => {
            if (gameState === "PLAYING" && touchX !== null) {
                touchX = e.clientX;
            }
        });

        window.addEventListener('touchmove', e => {
            if (gameState === "PLAYING") {
                e.preventDefault(); // Mencegah scroll
                touchX = e.touches[0].clientX;
            }
        }, { passive: false });

        window.addEventListener('mouseup', () => touchX = null);
        window.addEventListener('touchend', () => touchX = null);


        // ==========================================
        // 3. KELAS OBJEK (Ship, Asteroid, Particle)
        // ==========================================

        class Ship {
            constructor() {
                this.width = 60;
                this.height = 80;
                this.x = width / 2;
                this.y = height - 120;
                this.speed = 7;
                this.color = "#38bdf8"; // Sky Blue
            }

            update() {
                // Keyboard Movement
                if (keys.left && this.x > this.width / 2) this.x -= this.speed;
                if (keys.right && this.x < width - this.width / 2) this.x += this.speed;

                // Touch/Mouse Follow Movement (Simple lerp)
                if (touchX !== null) {
                    let dx = touchX - this.x;
                    this.x += dx * 0.1; // Smooth follow

                    // Batas layar
                    if (this.x < this.width / 2) this.x = this.width / 2;
                    if (this.x > width - this.width / 2) this.x = width - this.width / 2;
                }
            }

            draw() {
                ctx.fillStyle = this.color;

                // Badan Pesawat (Segitiga Sederhana)
                ctx.beginPath();
                ctx.moveTo(this.x, this.y);
                ctx.lineTo(this.x - this.width / 2, this.y + this.height);
                ctx.lineTo(this.x + this.width / 2, this.y + this.height);
                ctx.closePath();
                ctx.fill();

                // Mesin (Api)
                ctx.fillStyle = "#f59e0b";
                ctx.beginPath();
                ctx.moveTo(this.x - 10, this.y + this.height);
                ctx.lineTo(this.x + 10, this.y + this.height);
                ctx.lineTo(this.x, this.y + this.height + (Math.random() * 20 + 10));
                ctx.closePath();
                ctx.fill();
            }
        }

        class Asteroid {
            constructor(isCorrect, value) {
                this.r = 35; // Radius
                this.x = Math.random() * (width - this.r * 2) + this.r;
                this.y = -50;

                // Kecepatan Jatuh (Meningkat tiap level)
                // Level 1 sangat pelan agar mudah dibaca
                this.baseSpeed = (Math.random() * 0.5 + 0.5) + (level * 0.2);
                this.speed = this.baseSpeed;

                this.isCorrect = isCorrect;
                this.value = value;
                this.angle = 0;
                this.spinSpeed = Math.random() * 0.1 - 0.05;

                this.color = "#94a3b8"; // Slate 400
            }

            update() {
                // FITUR JATUH CEPAT (FAST DROP)
                let currentSpeed = this.baseSpeed;
                if (keys.space && gameState === "PLAYING") {
                    currentSpeed += 15;
                }

                this.y += currentSpeed;
                this.angle += this.spinSpeed;
            }

            draw() {
                ctx.save();
                ctx.translate(this.x, this.y);
                ctx.rotate(this.angle);

                // Gambar Batu
                ctx.fillStyle = this.color;
                ctx.beginPath();
                ctx.arc(0, 0, this.r, 0, Math.PI * 2);
                ctx.fill();

                // Detail Batu
                ctx.fillStyle = "rgba(0,0,0,0.2)";
                ctx.beginPath();
                ctx.arc(-10, -10, 8, 0, Math.PI * 2);
                ctx.fill();

                ctx.restore();

                // Teks Angka
                ctx.fillStyle = "#fff";
                ctx.font = "bold 20px Arial";
                ctx.textAlign = "center";
                ctx.textBaseline = "middle";
                ctx.fillText(this.value, this.x, this.y);
            }
        }

        class Particle {
            constructor(x, y, color) {
                this.x = x;
                this.y = y;
                this.vx = (Math.random() - 0.5) * 10;
                this.vy = (Math.random() - 0.5) * 10;
                this.life = 1.0;
                this.color = color;
            }
            update() {
                this.x += this.vx;
                this.y += this.vy;
                this.life -= 0.05;
            }
            draw() {
                ctx.globalAlpha = this.life;
                ctx.fillStyle = this.color;
                ctx.beginPath();
                ctx.arc(this.x, this.y, 3, 0, Math.PI * 2);
                ctx.fill();
                ctx.globalAlpha = 1.0;
            }
        }

        // ==========================================
        // 4. LOGIKA GAME & SOAL MATEMATIKA
        // ==========================================

        async function fetchQuestions() {
            const urlParams = new URLSearchParams(window.location.search);
            const gameId = urlParams.get('game_id') || 2;
            try {
                const res = await fetch(`questions.php?game_id=${gameId}`);
                if (res.ok) {
                    const data = await res.json();
                    if (data && Object.keys(data).length > 0) {
                        dbQuestions = data;
                        console.log("Questions loaded:", dbQuestions);
                    }
                }
            } catch (e) {
                console.error("Failed to fetch questions:", e);
            }
        }

        function generateQuestion() {
            // Cek apakah ada soal dari DB untuk level ini
            // Kita mapping level game (unlimited) ke level soal (1-3)
            // Level 1-5 -> Soal DB Level 1
            // Level 6-10 -> Soal DB Level 2
            // Level 11+ -> Soal DB Level 3
            // Atau cycle: (level - 1) % 3 + 1
            
            const dbLevel = ((level - 1) % 3) + 1;
            
            if (dbQuestions[dbLevel] && dbQuestions[dbLevel].q) {
                const qData = dbQuestions[dbLevel];
                return {
                    text: qData.q,
                    answer: qData.options[qData.correct], // Jawaban benar (teks)
                    options: qData.options, // Semua pilihan
                    isCustom: true
                };
            }

            // Fallback: Soal Penjumlahan/Pengurangan Sederhana Bilangan Bulat
            let min = -5 * level;
            let max = 10 * level;

            let a = Math.floor(Math.random() * (max - min + 1)) + min;
            let b = Math.floor(Math.random() * (max - min + 1)) + min;
            let operator = Math.random() > 0.5 ? '+' : '-';

            let result;
            if (operator === '+') result = a + b;
            else result = a - b;

            let text = `${a} ${operator} ${b >= 0 ? b : '(' + b + ')'} = ?`;

            return {
                text: text,
                answer: result,
                isCustom: false
            };
        }

        function spawnAsteroids() {
            if (gameState !== "PLAYING") return;

            // Bersihkan asteroid lama jika kosong
            if (asteroids.length === 0) {
                let pendingAsteroids = [];

                if (currentQuestion.isCustom) {
                    // --- LOGIKA SOAL CUSTOM (DB) ---
                    // Spawn asteroid sesuai pilihan jawaban (A, B, C)
                    const options = currentQuestion.options;
                    const correctAnswer = currentQuestion.answer;
                    
                    // Spawn semua opsi sebagai asteroid
                    // Pastikan posisi tidak tumpang tindih
                    const minDistance = 110;
                    
                    for (let opt of options) {
                        let isCorrect = (opt === correctAnswer);
                        let ast = new Asteroid(isCorrect, opt);
                        
                        let validPosition = false;
                        let attempts = 0;

                        while (!validPosition && attempts < 100) {
                            let testX = Math.random() * (width - ast.r * 4) + ast.r * 2;
                            let overlap = false;

                            for (let existing of pendingAsteroids) {
                                if (Math.abs(existing.x - testX) < minDistance) {
                                    overlap = true;
                                    break;
                                }
                            }

                            if (!overlap) {
                                ast.x = testX;
                                validPosition = true;
                            }
                            attempts++;
                        }
                        pendingAsteroids.push(ast);
                    }
                    
                } else {
                    // --- LOGIKA SOAL RANDOM (FALLBACK) ---
                    // 1. Tentukan Jumlah Asteroid
                    let maxTotal = width < 600 ? 3 : 5;
                    let wrongCount = (2 + Math.floor(level / 2));

                    if (wrongCount + 1 > maxTotal) wrongCount = maxTotal - 1;

                    // 2. Siapkan Data Jawaban
                    let answersData = [];

                    answersData.push({
                        val: currentQuestion.answer,
                        isCorrect: true
                    });

                    for (let i = 0; i < wrongCount; i++) {
                        let wrongAns;
                        let safety = 0;

                        do {
                            wrongAns = currentQuestion.answer + Math.floor(Math.random() * 20) - 10;
                            safety++;
                        } while ((wrongAns === currentQuestion.answer || answersData.some(a => a.val === wrongAns)) && safety < 50);

                        answersData.push({
                            val: wrongAns,
                            isCorrect: false
                        });
                    }

                    // 3. Buat Objek Asteroid dengan Posisi Anti-Overlap
                    const minDistance = 110;

                    for (let data of answersData) {
                        let ast = new Asteroid(data.isCorrect, data.val);
                        let validPosition = false;
                        let attempts = 0;

                        while (!validPosition && attempts < 100) {
                            let testX = Math.random() * (width - ast.r * 4) + ast.r * 2;
                            let overlap = false;

                            for (let existing of pendingAsteroids) {
                                if (Math.abs(existing.x - testX) < minDistance) {
                                    overlap = true;
                                    break;
                                }
                            }

                            if (!overlap) {
                                ast.x = testX;
                                validPosition = true;
                            }

                            attempts++;
                        }

                        pendingAsteroids.push(ast);
                    }
                }

                asteroids = pendingAsteroids;
            }
        }

        function createExplosion(x, y, color) {
            for (let i = 0; i < 15; i++) {
                particles.push(new Particle(x, y, color));
            }
        }

        function resetGame() {
            score = 0;
            lives = 3;
            level = 1;
            ship = new Ship();
            asteroids = [];
            particles = [];
            currentQuestion = generateQuestion();
            gameState = "PLAYING";

            spawnAsteroids();
        }

        function gameOver() {
            gameState = "GAMEOVER";
        }

        function gameWin() {
            gameState = "VICTORY";
            // Efek perayaan kemenangan
            createExplosion(width / 2, height / 2, "#4ade80");
            createExplosion(width / 4, height / 3, "#facc15");
            createExplosion(width * 0.75, height / 3, "#60a5fa");
        }

        // ==========================================
        // 5. GAME LOOP UTAMA
        // ==========================================

        function loop() {
            // Clear Screen
            ctx.fillStyle = "#0f172a";
            ctx.fillRect(0, 0, width, height);

            // Bintang Latar
            ctx.fillStyle = "#ffffff";
            for (let i = 0; i < 5; i++) {
                let sx = Math.random() * width;
                let sy = Math.random() * height;
                ctx.fillRect(sx, sy, 2, 2);
            }

            if (gameState === "PLAYING") {
                // Update Ship
                ship.update();
                ship.draw();

                // Spawn Asteroids logic
                if (asteroids.length === 0) spawnAsteroids();

                // Update Asteroids
                for (let i = asteroids.length - 1; i >= 0; i--) {
                    let a = asteroids[i];
                    a.update();
                    a.draw();

                    // Deteksi Tabrakan
                    let dist = Math.hypot(a.x - ship.x, a.y - (ship.y + ship.height / 2));

                    if (dist < a.r + ship.width / 2) {
                        // Tabrakan!
                        createExplosion(a.x, a.y, a.isCorrect ? "#4CAF50" : "#ef4444");

                        if (a.isCorrect) {
                            score += 10;
                            if (score % 50 === 0) level++;

                            // ===== CEK KONDISI MENANG DI SINI =====
                            if (score >= 150) {
                                gameWin();
                                return; // Keluar dari loop update ini
                            }

                            currentQuestion = generateQuestion();
                            asteroids = [];
                            createExplosion(ship.x, ship.y, "#ffffff");
                        } else {
                            lives--;
                            asteroids.splice(i, 1);
                            if (lives <= 0) gameOver();
                        }
                        continue;
                    }

                    // Hapus jika lewat bawah layar
                    if (a.y > height + 50) {
                        asteroids.splice(i, 1);

                        if (a.isCorrect) {
                            lives--;
                            currentQuestion = generateQuestion();
                            asteroids = [];
                            if (lives <= 0) gameOver();
                        }
                    }
                }

                // Update Particles
                for (let i = particles.length - 1; i >= 0; i--) {
                    particles[i].update();
                    particles[i].draw();
                    if (particles[i].life <= 0) particles.splice(i, 1);
                }

                // Draw HUD
                drawHUD();

            } else if (gameState === "START") {
                drawMenu("Misi Bilangan Bulat", "Klik Layar / Tekan Tombol untuk Mulai", "#38bdf8");
            } else if (gameState === "GAMEOVER") {
                drawMenu("Game Over", `Skor Akhir: ${score}\n\n- Klik untuk Coba Lagi`, "#ef4444");
            } else if (gameState === "VICTORY") {
                // Update Particles (Kembang Api di layar Victory)
                for (let i = particles.length - 1; i >= 0; i--) {
                    particles[i].update();
                    particles[i].draw();
                    if (particles[i].life <= 0) particles.splice(i, 1);
                }
                drawMenu("MISI SUKSES!", "Luar Angkasa Telah Aman! (Skor: 150)", "#4ade80");
            }

            requestAnimationFrame(loop);
        }

        function drawHUD() {
            // Latar HUD Atas
            ctx.fillStyle = "rgba(15, 23, 42, 0.8)";
            ctx.fillRect(0, 0, width, 80);
            ctx.strokeStyle = "#334155";
            ctx.lineWidth = 2;
            ctx.beginPath();
            ctx.moveTo(0, 80);
            ctx.lineTo(width, 80);
            ctx.stroke();

            // Soal
            ctx.fillStyle = "#fbbf24"; // Amber
            ctx.font = "bold 30px Arial";
            ctx.textAlign = "center";
            ctx.fillText(currentQuestion.text, width / 2, 50);

            // Skor & Nyawa & Level
            ctx.fillStyle = "#fff";
            ctx.font = "20px Arial";
            ctx.textAlign = "left";

            ctx.fillText(`Skor: ${score}`, 20, 30);
            ctx.font = "14px Arial";
            ctx.fillStyle = "#94a3b8";

            ctx.fillText(`Level: ${level}`, 20, 55);

            ctx.textAlign = "right";
            ctx.font = "20px Arial";
            ctx.fillStyle = lives > 1 ? "#fff" : "#ff4444";

            ctx.fillText(`Nyawa: ${lives}`, width - 20, 40);

            // Petunjuk Tombol Spasi
            ctx.font = "italic 14px Arial";
            ctx.fillStyle = "rgba(255, 255, 255, 0.5)";
            ctx.textAlign = "center";
            ctx.fillText("[ Tahan SPASI untuk jatuh cepat ]", width / 2, height - 30);
        }

        function drawMenu(title, subtitle, color) {
            ctx.fillStyle = "rgba(0,0,0,0.8)";
            ctx.fillRect(0, 0, width, height);

            ctx.fillStyle = color;
            ctx.font = "bold 40px Arial";
            ctx.textAlign = "center";
            ctx.fillText(title, width / 2, height / 2 - 20);

            ctx.fillStyle = "#fff";
            ctx.font = "20px Arial";
            // Handle multiline subtitle manually if needed, or just simple
            // For now simple fillText
            const lines = subtitle.split('\n');
            for (let i = 0; i < lines.length; i++) {
                ctx.fillText(lines[i], width / 2, height / 2 + 30 + (i * 30));
            }
        }

        // Tutorial Start Button Handler
        const startBtn = document.getElementById('start-game-btn');
        const tutorialOverlay = document.getElementById('tutorial-overlay');
        
        // Game starts in PAUSED state until tutorial is dismissed
        gameState = "PAUSED";
        
        if (startBtn && tutorialOverlay) {
            startBtn.addEventListener('click', () => {
                tutorialOverlay.style.display = 'none';
                resetGame(); // This sets gameState to PLAYING
            });
        }

        // Init Data
        fetchQuestions().then(() => {
            // Start Loop
            loop();
        });
        
    </script>
</body>

</html>


